<?php
/**
 * Class WC_Email_Customer_ACH_In_Process_Order file.
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! class_exists( 'WC_Email_Customer_ACH_In_Process_Order', false ) ) :

	/**
	 * Customer ACH In Process Order Email.
	 *
	 * An email sent to the customer when a new order is ACH In Process for.
	 *
	 * @class       WC_Email_Customer_ACH_In_Process_Order
	 * @extends     WC_Email
	 */
	class WC_Email_Customer_ACH_In_Process_Order extends WC_Email {

		/**
		 * Constructor.
		 */
		public function __construct() {
			$this->id             = 'customer_ach_in_process_order';
			$this->customer_email = true;
			$this->title          = __( 'ACH In Process order', 'paynote' );
			$this->description    = __( 'This is an order notification sent to customers containing order details after an order is placed ACH In Process.', 'paynote' );
			$this->template_html = 'templates/paynote-email-template.php';
		    $this->template_plain = 'templates/plain/paynote-email-template.php';
		    $this->template_base = PAYNOTE_WOO_GSP_DIR . '';
		    
			$this->placeholders   = array(
				'{order_date}'   => '',
				'{order_number}' => '',
			);

            //add_action( 'woocommerce_order_status_pachp-status', array( $this, 'trigger' ) );

			// Call parent constructor.
			parent::__construct();
		}

		/**
		 * Get email subject.
		 *
		 * @return string
		 */
		public function get_default_subject() {
			return __( '{site_title} - The status of your order #{order_number} has been updated to: ACH In Process', 'paynote' );
		}

		/**
		 * Get email heading.
		 *
		 * @return string
		 */
		public function get_default_heading() {
			return __( 'Your order #{order_number} is now in status: ACH In Process', 'paynote' );
		}

		/**
		 * Trigger the sending of this email.
		 *
		 * @param int            $order_id The order ID.
		 * @param WC_Order|false $order Order object.
		 */
		public function trigger( $order_id, $order = false ) {
			$this->setup_locale();

			if ( $order_id && ! is_a( $order, 'WC_Order' ) ) {
				$order = wc_get_order( $order_id );
			}

			if ( is_a( $order, 'WC_Order' ) ) {
				$this->object                         = $order;
				$this->recipient                      = $this->object->get_billing_email();
				$this->placeholders['{order_date}']   = wc_format_datetime( $this->object->get_date_created() );
				$this->placeholders['{order_number}'] = $this->object->get_order_number();
			}

			if ( $this->is_enabled() && $this->get_recipient() ) {
				$this->send( $this->get_recipient(), $this->get_subject(), $this->get_content(), $this->get_headers(), $this->get_attachments() );
			}

			$this->restore_locale();
		}

		/**
		 * Get content html.
		 *
		 * @return string
		 */
		public function get_content_html() {
			return wc_get_template_html(
				$this->template_html,
				array(
					'order'              => $this->object,
					'email_heading'      => $this->get_heading(),
					'additional_content' => $this->get_additional_content(),
					'sent_to_admin'      => false,
					'plain_text'         => false,
					'email'              => $this,
				), '', $this->template_base
			);
		}

		/**
		 * Get content plain.
		 *
		 * @return string
		 */
		public function get_content_plain() {
			return wc_get_template_html(
				$this->template_plain,
				array(
					'order'              => $this->object,
					'email_heading'      => $this->get_heading(),
					'additional_content' => $this->get_additional_content(),
					'sent_to_admin'      => false,
					'plain_text'         => true,
					'email'              => $this,
				), '', $this->template_base
			);
		}

		/**
		 * Default content to show below main email content.
		 *
		 * @return string
		 */
		public function get_default_additional_content() {
			return __( '<p>Verification has been completed, and the payment is currently in process. Once the payment is processed, you will receive a notification with the updated status.</p><p>For more details, please contact the store.</p>', 'paynote' );
		}

		/**
		 * Initialise Settings Form Fields - these are generic email options most will use.
		 */
		public function init_form_fields() {
			/* translators: %s: list of placeholders */
			$placeholder_text  = sprintf( __( 'Available placeholders: %s', 'paynote' ), '<code>' . esc_html( implode( '</code>, <code>', array_keys( $this->placeholders ) ) ) . '</code>' );
			$this->form_fields = array(
				'enabled'            => array(
					'title'   => __( 'Enable/Disable', 'paynote' ),
					'type'    => 'checkbox',
					'label'   => __( 'Enable this email notification', 'paynote' ),
					'default' => 'no',
				),
				'subject'            => array(
					'title'       => __( 'Subject', 'paynote' ),
					'type'        => 'text',
					'desc_tip'    => true,
					'description' => $placeholder_text,
					'placeholder' => $this->get_default_subject(),
					'default'     => '',
				),
				'heading'            => array(
					'title'       => __( 'Email heading', 'paynote' ),
					'type'        => 'text',
					'desc_tip'    => true,
					'description' => $placeholder_text,
					'placeholder' => $this->get_default_heading(),
					'default'     => '',
				),
				'additional_content' => array(
					'title'       => __( 'Additional content', 'paynote' ),
					'description' => __( 'Text to appear below the main email content.', 'paynote' ) . ' ' . $placeholder_text,
					'css'         => 'width:400px; height: 75px;',
					'placeholder' => __( 'N/A', 'paynote' ),
					'type'        => 'textarea',
					'default'     => $this->get_default_additional_content(),
					'desc_tip'    => true,
				),
				'email_type'         => array(
					'title'       => __( 'Email type', 'paynote' ),
					'type'        => 'select',
					'description' => __( 'Choose which format of email to send.', 'paynote' ),
					'default'     => 'html',
					'class'       => 'email_type wc-enhanced-select',
					'options'     => $this->get_email_type_options(),
					'desc_tip'    => true,
				),
			);
		}
	}

endif;

return new WC_Email_Customer_ACH_In_Process_Order();
