<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

if (!class_exists('WC_Gateway_Paynote')) {

    class WC_Gateway_Paynote extends WC_Payment_Gateway {

        /** @var bool Whether or not logging is enabled */
        public static $log_enabled = false;

        /** @var WC_Logger Logger instance */
        public static $log = false;
        public static $deleted;
        public static $processed;
        public static $rejected;

        /**
         * @var string The last error to have occurred on the gateway
         */
        public $error = "";

        /**
         * @var array An array of options set by the gateway settings page recalled by get_option()
         */
        public $options = array();

        /**
         * @var string The current API endpoint URL to be called which will be appended by the method.
         */
        public $endpoint = PAYNOTE_ENDPOINT_LINK_LIVE;

        /**
         * @var string The current API endpoint to be called as far as eCheck, eCart, etc.
         */
        public $method = "WooCommerce.asmx";

        /**
         * @var string The merchant's API Client ID
         */
        public $client_id = false;

        /**
         * @var string The merchant's API Password
         */
        public $live_api = false;

        /**
         * @var string The woocommerce store REST API client ID
         */
        public $rest_client_id = false;

        /**
         * @var string The woocommerce store REST API client secret
         */
        public $rest_client_secret = false;

        /**
         * @var bool Whether or not the store allows the widget to be displayed on the front end
         */
        public $allow_widget = false;

        /**
         * @var string A human readable description of the payment method displayed on the checkout page
         */
        public $description = "";

        /**
         * @var string any extra text to add to the description of the payment method on the checkout page
         */
        public $extra = "";

        /**
         * @var string
         */
        public $cronPeriod = 0;

        /**
         * @var bool True if the plugin should be logging requests more verbosely to a log file found in wp-content/uploads/wc-logs/ by default
         */
        public $debug = false;

        /**
         * @var string Will either be set to the user input setting API URL or will be get_site_url()
         */
        public $useStoreURL = "";

        /**
         * Cloning is forbidden
         *
         * @since 1.2.0
         */
        public function __clone() {
            //do nothing
        }

        /**
         * Unserializing instances of this class is forbidden
         *
         * @since 1.2.0
         */
        public function __wakeup() {
            //do nothing
        }

        public $if_recurring;
        public $recurring_installments;
        public $recurring_cycle;
        public $is_subscription;
        public $recurring;
        public $recurring_start_date;
        public $is_save_bank_details;
        public $version_api;
        public $verification_mode;
        public $paynote_check_id;
        // public $save_bank_details_user_id;
        // public $save_bank_details_user_email;
        public $id;

        //See parent class WC_Payment_Gateway for (id, has_fields, method_title, method_description, title, supports)
        public function __construct() {
            $this->error = '';
            $this->id = 'paynote';
            $this->has_fields = true;
            $this->method_title = esc_html__('Paynote', 'paynote');
            $new_settings_page = get_admin_url(null, 'admin.php?page=paynote_gateway');
            /* translators: %s - location plugin settings. */
            $this->method_description = sprintf(__("<a href='%s' target='_blank'>Take payments in Paynote with fast next day deposits directly to your bank account.</a>", 'paynote'), esc_attr($new_settings_page));
            $this->supports = array(
                'products',
                'refunds',
                'tokenization'
            );

            //Get options we need and make them usable
            $this->options = get_option('paynote_settings');
            $this->client_id = (isset($this->options['paynote_test_api_key'])) ? $this->options['paynote_test_api_key'] : false;
            $this->live_api = (isset($this->options['paynote_live_api_key'])) ? $this->options['paynote_live_api_key'] : false;

            $this->endpoint = (isset($this->options['paynote_api_endpoint'])) ? trailingslashit($this->options['paynote_api_endpoint']) : PAYNOTE_ENDPOINT_LINK_LIVE;
            
            // $this->rest_client_id = (isset($this->options['paynote_woo_rest_client_id'])) ? $this->options['paynote_woo_rest_client_id'] : false;
            // $this->rest_client_secret = (isset($this->options['paynote_woo_rest_client_secret'])) ? $this->options['paynote_woo_rest_client_secret'] : false;
            $this->allow_widget = (isset($this->options['paynote_tokenization_use'])) ? ($this->options["paynote_tokenization_use"] == 1) : false;

            $this->extra = (isset($this->options['paynote_extra_message'])) ? $this->options['paynote_extra_message'] : "";
            $this->if_recurring = (isset($this->options['paynote_gateway_recurring'])) ? $this->options['paynote_gateway_recurring'] : "month";
            $this->recurring_installments = (isset($this->options['paynote_recurring_installments'])) ? $this->options['paynote_recurring_installments'] : "0";
            $this->recurring_cycle = (isset($this->options['paynote_recurring_cycle'])) ? $this->options['paynote_recurring_cycle'] : "";
            $this->is_subscription = $this->if_recurring == 1 ? 1 : 0;
            $this->recurring = $this->if_recurring != 0 ? 1 : 0;
            $this->recurring_start_date = null;

            if ($this->is_subscription) {
                $this->recurring_installments = 0;
                $this->recurring_cycle = 'month';
            }

            $this->is_save_bank_details = (isset($this->options['paynote_save_bank_details'])) ? $this->options['paynote_save_bank_details'] : 0;
            $this->version_api = PAYNOTE_VERSION_API ? PAYNOTE_VERSION_API : 'v1';
            $this->description = (isset($this->options['paynote_gateway_description'])) ? $this->options['paynote_gateway_description'] : "";
            $this->verification_mode = (isset($this->options['paynote_override_risky_option'])) ? $this->options['paynote_override_risky_option'] : "legacy";
            $this->cronPeriod = (isset($this->options['paynote_settings_order_cron'])) ? $this->options['paynote_settings_order_cron'] : 0;
            $this->title = (isset($this->options['paynote_title'])) ? $this->options['paynote_title'] : "";
            $this->useStoreURL = get_site_url();
            $this->debug = TRUE;
            self::$log_enabled = $this->debug;
            $this->enabled = $this->get_option( 'enabled' );

            // Method with all the options fields
            $this->init_form_fields();

            add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
            //add_action('admin_notices', array($this, 'payment_do_ssl_check'), 999);

            // We may need to add in the custom JS for the widget. $this->payment_scripts will run on wp_enqueue_scripts to determine if that's the case and potentially inject the code
            add_action('wp_enqueue_scripts', array($this, 'payment_scripts'));
            add_action('woocommerce_cancelled_order', 'order_cancelled_so_cancelcheck', 10, 1);
        }

        function __toString() {
            $str = "Gateway Type: POST\n";
            $str .= "Endpoint: " . $this->endpoint . "\n";
            $str .= "Client ID: " . $this->client_id . "\n";
            $str .= "ApiPassword: " . $this->live_api . "\n";

            return $str;
        }

        public function init_form_fields(){

            $this->form_fields = array(
                'enabled'         => array(
                'title'       => __( 'Enable/Disable', 'paynote' ),
                'label'       => __( 'Enable Paynote Payment', 'paynote' ),
                'type'        => 'checkbox',
                'description' => '',
                'default'     => 'yes',
                )
            );
        }

        /**
         * Internal helper function returns the entire endpoint URL
         *
         * @return string The full unqualified URL an API call is targeted to for this Gateway
         */
        public function full_endpoint() {
            return $this->endpoint . $this->version_api . $this->method;
        }

        function paynote_toString($html = TRUE) {
            if ($html) {
                return nl2br($this->__toString());
            }

            return $this->__toString();
        }

        private function paynote_setLastError($error) {
            $this->error = $error;
        }

        public function paynote_getLastError() {
            return $this->error;
        }

        /**
         * Logging method.
         *
         * @param string $message
         */
        public static function log($message) {
            if (self::$log_enabled) {
                if (empty(self::$log)) {
                    self::$log = new WC_Logger();
                }
                self::$log->add('paynote', $message);
            }
        }

        // Check if we are forcing SSL on checkout pages
        public function payment_do_ssl_check() {
            if (( function_exists('wc_site_is_https') && !wc_site_is_https() ) && ( 'no' === get_option('woocommerce_force_ssl_checkout') && !class_exists('WordPressHTTPS') )) {
                /* translators: %1$s: title, %2$s: location page settings checkout. */
                echo wp_kses_post('<div class="error"><p>' . sprintf(__('<strong>%1$s</strong> is enabled and WooCommerce is not forcing the SSL certificate on your checkout page. Please ensure that you have a valid SSL certificate and that you are <a href="%2$s">forcing the checkout pages to be secured.</a>', 'paynote'), esc_html($this->method_title), esc_url(admin_url('admin.php?page=wc-settings&tab=checkout'))) . '</p></div>');
            }
        }

        

        /**
         * The order was cancelled so we want to try and cancel the Check in Paynote
         *
         * @param string $order_id
         */
        public function order_cancelled_so_cancelcheck($order_id) {

            $order = wc_get_order($order_id);

            if(!$order || $order->get_payment_method() != 'paynote'){
                return;
            }

            $order_status =  $order->get_status();            
            $paynote_check_id = $order->get_meta('paynote_check_id');
            $check = $this->get_check_by_paynote_check_id($paynote_check_id);

            if ($check) {
                $cancelled = in_array(strtolower($check->status), ["voided", "cancelled"]);
                $pending = in_array(strtolower($check->status), ["pending"]);
                $processed = in_array(strtolower($check->status), ["printed", "processed"]);
                $refunded = in_array(strtolower($check->status), ["refunded"]);
                $failed = in_array(strtolower($check->status), ["failed", "expired", "refund failed"]);
                $ach_processing = in_array(strtolower($check->status), ["processing"]);
                $ach_declined = in_array(strtolower($check->status), ["declined"]);
                $ach_hold = in_array(strtolower($check->status), ["hold"]);

                if (!$cancelled) {
                    $response = $this->cancelled_check_by_paynote_check_id($paynote_check_id);
                    if ($response->success) {
                        /* translators: %s - paynote check id. */
                        $order->add_order_note(sprintf(__('Order Cancelled so Check was also Cancelled #%s', 'paynote'), $paynote_check_id ));
                    } else if ($response->error && $response->message) {

                        /* translators: %1$s: paynote check id, %2$s: response message. */
                        $order->add_order_note(sprintf(__('Error order Cancelled so Check: %1$s . %2$s', 'paynote'), $paynote_check_id, $response->message));

                        if ($refunded) {
                            $order->update_status('refunded');
                        } else if ($processed) {
                            $order->update_status('processing');
                        } else if ($failed) {
                            $order->update_status('failed');
                        } else if ($ach_processing) {
                            $order->update_status('pachpr-status');
                        } else if ($ach_declined) {
                            $order->update_status('pachd-status');
                        } else if ($ach_hold) {
                            $order->update_status('pachoh-status');
                        } else {

                            if ($order_status != 'on-hold' && $order_status != 'pachp-status'){

                                if ($options['paynote_settings_default_order_status'] == 'on-hold') {
                                    $order->add_order_note(__('Verification process completed by Paynote and check is in queue to be processed. Once the check has processed at Paynote, we will update your order status from On-Hold to Processing.', 'paynote')); 
                                    $order->update_status('on-hold');
                                }

                                if ($options['paynote_settings_default_order_status'] == 'ach-in-process') {

                                    $order->add_order_note(__('Verification process completed by Paynote and check is in queue to be processed. Once the check has processed at Paynote, we will update your order status from ACH In Process to Processing.', 'paynote'));
                                    $order->update_status('pachp-status');
                                }
                            }                         
                            
                        }

                    } 
                    
                }
            }
        }

        /**
         * Check if this gateway is enabled
         */
        public function is_available() {
           return 'yes' === $this->enabled && parent::is_available();
        }

        /**
         * Payment form on checkout page.
         *
         * See WC_Payment_Gateway::payment_fields()
         */
        public function payment_fields() {
            include('manual-enrollment.php');
        }

        /**
         * Safely get and trim data from $_POST
         *
         * @since 1.2.0
         * @param string $key array key to get from $_POST array
         * @return string value from $_POST or blank string if $_POST[ $key ] is not set
         */
        public static function get_post($key) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing         
            if (isset($_POST[$key])) {
                // phpcs:ignore WordPress.Security.NonceVerification.Missing
                return sanitize_text_field(wp_unslash($_POST[$key])) ;
            }
            return '';
        }

        /**
         * Validate the payment fields when processing the checkout
         *
         * @since 1.2.0
         * @see WC_Payment_Gateway::validate_fields()
         * @return bool true if fields are valid, false otherwise
         */
        public function validate_fields() {
            $is_valid = parent::validate_fields(); //$is_valid = true in MOST cases
            return $this->validate_paynote_fields($is_valid);
            return true;
        }

        /**
         * Returns true if the posted echeck paynote fields are valid, false otherwise
         *
         * @since 1.2.0
         * @param bool $is_valid true if the fields are valid, false otherwise
         * @return bool
         */
        protected function validate_paynote_fields($is_valid) {
            if (!$is_valid) {
                //form was invalid before it got to us. Immediately error out.
                return false;
            }

            return apply_filters('wc_payment_gateway_' . $this->id . '_validate_check_fields', $is_valid, $this);
        }

        function callAPI($messageName, $data, $method = null) {

            $this->method = $messageName;
                       
            $endpointId = $this->live_api;
            if ($this->endpoint == PAYNOTE_ENDPOINT_LINK_LIVE) {
                $endpointId = $this->live_api;
            }
            if ($this->endpoint == PAYNOTE_ENDPOINT_LINK_TEST) {
                $endpointId = $this->client_id;
            }

            $endpoint = $this->full_endpoint();
            
            $args = [
                'blocking' => true,
                'timeout' => 60,
                'headers' => [
                    'Authorization' => $endpointId
                ]
            ];

            $params = http_build_query($data);
            /* translators: %s - endpoint. */
            $this->log(sprintf(__('Preparing data, endpoint is: %s', 'paynote'), $endpoint));
            /* translators: %s - data params. */
            $this->log(sprintf(__('Sending POST to Paynote, this is what we are sending: %s', 'paynote'), $params));

            $response = NULL;

            if (strcasecmp(strtolower($method), "get") === 0) {
                $response = wp_remote_get($endpoint . ($params ? '?' . $params : ''), $args);
            } else if (strcasecmp(strtolower($method), "delete") === 0) {
                $args['method'] = 'DELETE';
                $response = wp_remote_request($endpoint . ($params ? '?' . $params : ''), $args);
            } else {
                $args['body'] = $data;
                $response = wp_remote_post($endpoint, $args);
            }

            if ( is_wp_error( $response ) ) {
                $error_message = $response->get_error_message();
                /* translators: %s - error message. */
                $this->log(sprintf(__('Error Response: %s', 'paynote'), $error_message));
                return  wp_json_encode(["error" => true, "message" => $error_message]);
            } 

            $body = wp_remote_retrieve_body($response);

            /* translators: %s - Response body. */
            $this->log(sprintf(__('Raw Response: %s', 'paynote'), $body));

            try {
                $loadedResponse = json_decode($body);
                return $loadedResponse;
            } catch (\Exception $e) {
                $this->log("Unable to parse API results: " . $e->getMessage());
                return wp_json_encode(["error" => true, "message" => '']);
            }
        }


        // Conditional function that check if Checkout page use Checkout Blocks
        public function is_checkout_block() {
            return WC_Blocks_Utils::has_block_in_page( wc_get_page_id('checkout'), 'woocommerce/checkout' );
        }

        /**
         *
         * @param int $order_id
         * @return array
         *
         */
        public function process_payment($order_id) {            
            // phpcs:ignore WordPress.Security.NonceVerification.Missing
            if (!isset( $_POST['paynote_account_id'], $_POST['paynote_public_token'] )
                // phpcs:ignore WordPress.Security.NonceVerification.Missing
                || sanitize_text_field(wp_unslash(!$_POST['paynote_account_id'])) || sanitize_text_field(wp_unslash(!$_POST['paynote_public_token'])))
            {
                /* translators: %1$s - INVALID REQUEST message, %2$s - Order ID. */
                $this->log(sprintf(__('INVALID_REQUEST public_token must be a non-empty string: %1$s Order ID: %2$s', 'paynote'), (isset($_SERVER["HTTP_USER_AGENT"]) ? sanitize_text_field(wp_unslash($_SERVER["HTTP_USER_AGENT"])) : ''), $order_id));
                throw new Exception('Select a bank account' );                
                //return null;
            }

            $order = wc_get_order($order_id);
            $orderdata = $this->get_order_info($order);
            if ('paynote' == $orderdata["payment_method"]) {
                /* translators: %s - order id. */
                $this->log(sprintf(__('Started to process order: #%s', 'paynote'), $order_id));
                return $this->process_payment_paynote($order_id, $order, $orderdata);
            }
            return null;
        }

        /**
         * Call to the Paynote API to generate the check and return the status of that to the front end
         *
         * @param int $order_id     The id of the WooCommerce Order instance we're checking out for.
         * @param mixed $order      The WooCommerce Order Instance we're checking out for.
         * @param array $orderdata  The result of self::get_order_info on the order
         *
         * @return array|void Returns an array with success info or null on error
         */
        private function process_payment_paynote($order_id, $order, $orderdata) {
            
            $data = [
                // phpcs:ignore WordPress.Security.NonceVerification.Missing
                'account_id' => (isset($_POST['paynote_account_id']) ? sanitize_text_field(wp_unslash($_POST['paynote_account_id'])) : ''),
                // phpcs:ignore WordPress.Security.NonceVerification.Missing
                'token' => (isset($_POST['paynote_public_token']) ? sanitize_text_field(wp_unslash($_POST['paynote_public_token'])) : ''),
                'firstName' => $orderdata['billing_first_name'],
                'lastName' => $orderdata['billing_last_name'],
                'email' => $orderdata['billing_email'],
                'phone' => $orderdata['billing_phone'],
                'businessName' =>  $orderdata['billing_company'],
                'capture_auth' => $this->is_save_bank_details
            ];

            $response = $this->callAPI("/user/express", $data, 'POST');

            if (isset($response->success) && isset($response->user)) {

                if ($this->is_save_bank_details) {
                    $order->payment_complete();
                    /* translators: %s - user email. */
                    $order->add_order_note(sprintf(__('<b>%s</b> has been authorized for future debits.', 'paynote'), $response->user->email));
                    $order->update_meta_data('paynote_save_bank_details_user_id', $response->user->user_id);
                    $order->update_meta_data('paynote_save_bank_details_user_email', $response->user->email);
                    $order->save();

                    return array(
                        'result' => 'success',
                        'redirect' => $this->get_return_url($order)
                    );
                } 

                $data = [
                    'amount' => $orderdata['order_total'],  
                    'description' => "Order #" . $order_id, 
                    'user_id' => $response->user->user_id, 
                    'identifier' => $this->useStoreURL . ', ' . $this->version_api .', '.PAYNOTE_VERSION
                ];

                if( $this->if_recurring ){
                    $data = array_merge($data, [
                        'recurring' => [
                            'recurring' => 1,
                            'billing_cycle' => $this->recurring_cycle,
                            'start_cycle' => gmdate('Y-m-d'),
                            'num_of_payments' => $this->recurring_installments
                        ]
                    ]);
                }

                $response = $this->callAPI("/check/express", $data, 'POST');

                if (isset($response->success) && $response->success && $response->check && $response->check->status == 'pending') {

                    if (isset($response->check->sndr_name)) {
                        $name_parts = explode(' ', $response->check->sndr_name, 2);

                        if (empty($order->get_billing_first_name()) && !empty($name_parts[0])) {
                            $order->set_billing_first_name($name_parts[0]);
                        }

                        if (empty($order->get_billing_last_name()) && !empty($name_parts[1])) {
                            $order->set_billing_last_name($name_parts[1]);
                        }
                    }

                    if (isset($response->check->sndr_email) && empty($order->get_billing_email())) {
                        $order->set_billing_email($response->check->sndr_email);
                    }

                    /* translators: %1$s - Check ID, %2$s - Check Number. */
                    $this->log(sprintf(__('Paynote check accepted (Check_ID: %1$s, CheckNumber: %2$s)', 'paynote'),$response->check->check_id, $response->check->number));
                    if ($this->options['paynote_settings_default_order_status'] == 'on-hold') {
                        $order->add_order_note(__('Verification process completed by Paynote and check is in queue to be processed. Once the check has processed at Paynote, we will update your order status from On-Hold to Processing.', 'paynote'));
                        $order->update_status('on-hold', __('Awaiting cheque payment', 'paynote'));
                    }
                    if ($this->options['paynote_settings_default_order_status'] == 'ach-in-process') {
                        $order->add_order_note(__('Verification process completed by Paynote and check is in queue to be processed. Once the check has processed at Paynote, we will update your order status from ACH In Process to Processing.', 'paynote'));
                        $order->update_status('pachp-status', __('Awaiting cheque payment', 'paynote'));
                    }
                    /* translators: %1$s - Check ID, %2$s - Check Number. */
                    $order->add_order_note(sprintf(__('Paynote check accepted (Check_ID: %1$s, CheckNumber: %2$s)', 'paynote'), $response->check->check_id, $response->check->number));
                    $order->update_meta_data('paynote_check_id', $response->check->check_id );
                    $order->update_meta_data('paynote_payment_check_number', $response->check->number );
                    $order->save();  
                 
                    return array(
                        'result' => 'success',
                        'redirect' => $this->get_return_url($order)
                    );
                }
            }

            /* translators: %s - response message. */
            $this->log(sprintf(__('Check is not accepted. Paynote returned error description: %s', 'paynote'), isset($response, $response->message) ? $response->message : '')); 

            if (isset($response->check)) {
                $payalert = '';
                $order->add_order_note(sprintf(__('Paynote check accepted (Check_ID: %1$s, CheckNumber: %2$s)', 'paynote'), $response->check->check_id, $response->check->number));
                $order->update_meta_data('paynote_check_id', $response->check->check_id );
                $order->update_meta_data('paynote_payment_check_number', $response->check->number );
                if($response->check->status == 'declined'){
                    $order->add_order_note(sprintf(__('The transaction will be declined and not deposited due to - %s', 'paynote'), $response->check->error_description)); 
                    $order->update_status('pachd-status');
                    $payalert = __('<p>Payment for your order was declined.</p><p>For more details, please contact the store.</p>', 'paynote');
                }

                if($response->check->status == 'hold'){
                    $order->add_order_note(__('The transaction will be placed in a manual review queue and will only be deposited after your manual approval in the dashboard Paynote.', 'paynote')); 
                    $order->update_status('pachoh-status');
                    $payalert = __('<p>Your order has been successfully created, but the payment is currently under manual review by the store.</p><p>Once the store confirms the payment, your order will be processed, and you will receive a notification with the updated status.</p><p>For more details, please contact the store.</p>', 'paynote');
                }
                
                $order->save();

                return array(
                    'result' => 'success',
                    'redirect' => $this->paynote_get_return_url($order, $payalert)
                );
            }           


            throw new Exception(isset($response, $response->message) ? $response->message : '');
        }

        public function paynote_get_return_url( $order = null, $message = '' ) {
            if ( $order ) {
                $return_url = $order->get_checkout_order_received_url();
            } else {
                $return_url = wc_get_endpoint_url( 'order-received', '', wc_get_checkout_url() );
            }
            return apply_filters( 'woocommerce_get_return_url', $return_url.'&payalert='.urlencode($message), $order );
        }

        /**
         * Return the order information in a version independent way
         *
         * @param WC_Order $order
         * @return array
         */
        public function get_order_info($order) {
            $data = array(
                "id" => '',
                "payment_method" => '',
                "billing_company" => '',
                "billing_first_name" => '',
                "billing_last_name" => '',
                "billing_email" => '',
                "billing_phone" => '',
                "billing_address_1" => '',
                "billing_address_2" => '',
                "billing_city" => '',
                "billing_state" => '',
                "billing_postcode" => '',
                "billing_country" => '',
                "order_total" => '',
            );
            if (version_compare(WC_VERSION, '3.0', '<')) {
                //Do it the old school way
                $data["id"] = sanitize_text_field($order->id);
                $data["payment_method"] = sanitize_text_field($order->payment_method);
                $data["billing_company"] = sanitize_text_field($order->billing_company);
                $data["billing_first_name"] = sanitize_text_field($order->billing_first_name);
                $data["billing_last_name"] = sanitize_text_field($order->billing_last_name);
                $data["billing_email"] = sanitize_text_field($order->billing_email);
                $data["billing_phone"] = sanitize_text_field($order->billing_phone);
                $data["billing_address_1"] = sanitize_text_field($order->billing_address_1);
                $data["billing_address_2"] = sanitize_text_field($order->billing_address_2);
                $data["billing_city"] = sanitize_text_field($order->billing_city);
                $data["billing_state"] = sanitize_text_field($order->billing_state);
                $data["billing_postcode"] = sanitize_text_field($order->billing_postcode);
                $data["billing_country"] = sanitize_text_field($order->billing_country);
                $data["order_total"] = sanitize_text_field($order->order_total);
            } else {
                //New school
                $data["id"] = sanitize_text_field($order->get_id());
                $data["payment_method"] = sanitize_text_field($order->get_payment_method());
                $data["billing_company"] = sanitize_text_field($order->get_billing_company());
                $data["billing_first_name"] = sanitize_text_field($order->get_billing_first_name());
                $data["billing_last_name"] = sanitize_text_field($order->get_billing_last_name());
                $data["billing_email"] = sanitize_text_field($order->get_billing_email());
                $data["billing_phone"] = sanitize_text_field($order->get_billing_phone());
                $data["billing_address_1"] = sanitize_text_field($order->get_billing_address_1());
                $data["billing_address_2"] = sanitize_text_field($order->get_billing_address_2());
                $data["billing_city"] = sanitize_text_field($order->get_billing_city());
                $data["billing_state"] = sanitize_text_field($order->get_billing_state());
                $data["billing_postcode"] = sanitize_text_field($order->get_billing_postcode());
                $data["billing_country"] = sanitize_text_field($order->get_billing_country());
                $data["order_total"] = sanitize_text_field($order->get_total());
            }
            return $data;
        }

        public function get_check_by_paynote_check_id($check_id, $only_check = true) {
            $response = $this->callAPI("/check/:" . $check_id, [], 'GET');
            if ($only_check) {
                return isset($response->check) ? $response->check : null;
            } else {
                return $response;
            }
        }

        public function cancelled_check_by_paynote_check_id($check_id) {
            $response = $this->callAPI("/check/:" . $check_id, [], 'DELETE');
            return $response;
        }

        public function refund_check_by_paynote_check_id($check_id, $order_amount) {
            $response = $this->callAPI("/check/refund", ["check_id" => (string) $check_id, "amount" => $order_amount], 'POST');
            return $response;
        }

        public function get_webhook() {
            $response = $this->callAPI("/webhookurl", [], 'GET');
            return $response;
        }

        public function set_webhook($webhook_url) {
            $response = $this->callAPI("/webhookurl", ["webhook_url" => (string) $webhook_url], 'POST');
            return $response;
        }

        /**
         * @param  WC_Order $order
         * @return bool
         */
        public function via_paynote($order) {
            return $order && $order->get_meta('paynote_check_id');
        }

        /**
         * Process a refund if supported
         *
         * @param  int    $order_id
         * @param  float  $amount
         * @param  string $reason
         * @return  boolean True or false based on success, or a WP_Error object
         *
         * See WC_Payment_Gateway::process_refund()
         */
        public function process_refund($order_id, $amount = null, $reason = '') {

            $this->log(__('Started to process refund: #', 'paynote') . $order_id . ', $' . $amount ) ;

            $order = wc_get_order($order_id);
            $paynote_check_id = $order->get_meta('paynote_check_id');
            $paynote_payment_check_number = $order->get_meta('paynote_payment_check_number');
            $orderdata = $this->get_order_info($order);

            $order_status = $order->get_status();

            if ('paynote' == $orderdata["payment_method"]) {

                $refund_amount = $amount ?? number_format($order->get_total(), wc_get_price_decimals(), wc_get_price_decimal_separator(), wc_get_price_thousand_separator());

                if ($paynote_check_id) {

                    $check = $this->get_check_by_paynote_check_id($paynote_check_id);

                    if ($check) {

                        $cancelled = in_array(strtolower($check->status), ["voided", "cancelled"]);
                        $processed = in_array(strtolower($check->status), ["printed", "processed"]);
                        $refunded = in_array(strtolower($check->status), ["refunded", "refund pending"]);
                        $failed = in_array(strtolower($check->status), ["failed", "expired", "refund failed"]);
                        $pending = in_array(strtolower($check->status), ["pending"]);
                        $ach_processing = in_array(strtolower($check->status), ["processing"]);
                        $ach_declined = in_array(strtolower($check->status), ["declined"]);
                        $ach_hold = in_array(strtolower($check->status), ["hold"]);

                        if ($processed) {

                            $response = $this->refund_check_by_paynote_check_id($paynote_check_id, $refund_amount);

                            if ($response->success) {

                                /* translators: %s - Check ID. */
                                $this->log(sprintf(__('Refund Accepted. Paynote refund Check_ID: %s', 'paynote'), $check->check_id));

                                /* translators: %1$s - Check ID, %2$s - Check Number, %3$s - Check amount. */
                                $order->add_order_note(sprintf(__('Refund Accepted. Refund Paynote Check ID: %1$s, Paynote Check Number: %2$s, Check amount: $%3$s', 'paynote'), (string) $paynote_check_id, (string) $paynote_payment_check_number, (string) $refund_amount));
                                $order->update_status('refunded');
                                return true;
                            } else {
                                if ($response->error && $response->message) {
                                    /* translators: %s - Paynote message. */
                                    $order->add_order_note(sprintf(__('Paynote message: %s', 'paynote'), $response->message));
                                } else {
                                    $order->add_order_note(__('Paynote refund was declined', 'paynote'));
                                }
                                return false;
                            }
                        } else if ($cancelled) {
                            if ($order_status != 'cancelled') {
                                $order->update_status('cancelled');
                                $order->add_order_note(__('Unable to process a refund for this check. For a refund to be processed, this payment should be in "Processed" status. Please attempt again once the order is "Processed."', 'paynote'));
                            }
                            return true;
                        } else if ($refunded) {
                            if ($order_status != 'refunded') {
                                $order->update_status('refunded');
                                $order->add_order_note(__('Unable to process a refund for this check. For a refund to be processed, this payment should be in "Processed" status. Please attempt again once the order is "Processed."', 'paynote'));
                            }
                            return true;
                        } else if ($failed) {
                            if ($order_status != 'failed'){
                                $order->update_status('failed');
                                $order->add_order_note(__('Unable to process a refund for this check. For a refund to be processed, this payment should be in "Processed" status. Please attempt again once the order is "Processed."', 'paynote'));
                            }
                            return true;
                        } else if ($ach_processing) {
                            if ($order_status != 'pachpr-status'){
                                $order->update_status('pachpr-status');
                                $order->add_order_note(__('Unable to process a refund for this check. For a refund to be processed, this payment should be in "Processed" status. Please attempt again once the order is "Processed."', 'paynote'));
                            }
                            return true;
                        } else if ($ach_declined) {
                            if ($order_status != 'pachd-status'){
                                $order->update_status('pachd-status');
                                $order->add_order_note(__('Unable to process a refund for this check. For a refund to be processed, this payment should be in "Processed" status. Please attempt again once the order is "Processed."', 'paynote'));
                            }
                            return true;
                        } else if ($ach_hold) {
                            if ($order_status != 'pachoh-status'){
                                $order->update_status('pachoh-status');
                                $order->add_order_note(__('Unable to process a refund for this check. For a refund to be processed, this payment should be in "Processed" status. Please attempt again once the order is "Processed."', 'paynote'));
                            }
                            return true;
                        } else if ($pending) {
                            $order->add_order_note(__('Paynote Refund in process.', 'paynote'));
                            return true;
                        } else {

                            if ($order_status != 'on-hold' && $order_status != 'pachp-status'){

                                if ($options['paynote_settings_default_order_status'] == 'on-hold') {
                                    $order->add_order_note(__('Verification process completed by Paynote and check is in queue to be processed. Once the check has processed at Paynote, we will update your order status from On-Hold to Processing.', 'paynote')); 
                                    $order->update_status('on-hold');
                                }

                                if ($options['paynote_settings_default_order_status'] == 'ach-in-process') {

                                    $order->add_order_note(__('Verification process completed by Paynote and check is in queue to be processed. Once the check has processed at Paynote, we will update your order status from ACH In Process to Processing.', 'paynote'));
                                    $order->update_status('pachp-status');
                                }
                            }
                            return true;
                        }
                    } else {
                        $this->log(__('Unable to process a refund for this check.', 'paynote'));
                    }
                } else {
                    $this->log(__('Not Paynote check ID', 'paynote'));
                }
            }
            return false;
        }

        /**
         * Call the Test API's TestAuthentication to validate the merchant's API credentials with their selected endpoint.
         *
         * @return boolean  True if the credentials validate, false otherwise.
         */
        public function test_authentication() {

            return true;
        }

        /**
         * A function to determine whether the current Paynote client can use the tokenization widget
         *
         * @return boolean True if they can use the widget. False if not.
         */
        public function can_widget() {
            return FALSE;
        }

        /**
         * Will call to Paynote to register the current store's WooCommerce REST API credentials
         *
         * @return boolean True if the store was/is saved, false if some other error occurred.
         */
        public function register_store($storeURL) {

            return true;
        }

        /**
         * Function will make an API call to our API that will register the session in our server
         *
         * @return boolean True if they can use the widget. False if not.
         */
        public function start_session($sessionId) {
            return true;
        }

        /**
         * Add payment JS script for tokenization if necessary
         */
        public function payment_scripts() {

            if (
                !is_cart() && 
                !is_checkout() &&                 
                !isset(
                    // phpcs:ignore WordPress.Security.NonceVerification.Recommended
                    $_GET['pay_for_order']
                )
            ) {
                return;
            }
        }

    }

} 
