<?php

/**
 * This filter ads the Paynote Status Update All button to the top of WooCommerce Orders page
 */

/**
 * @param  string $hook
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

function paynote_js_enqueue($hook) {
    wp_enqueue_script('status-update-script', plugins_url('../js/paynote_scripts.js', __FILE__), array('jquery'), PAYNOTE_VERSION, true );
    wp_localize_script('status-update-script', 'ajax_object_name', array(
        'ajax_url' => admin_url('admin-ajax.php'),
        'security' => wp_create_nonce("status_update_nonce"),
        'expected_webhook_url' => get_site_url() . '/wc-api/paynote_webhook/'
    ));
}

/*******************************************************/
add_action("wp_ajax_paynote_set_webhook", "paynote_set_webhook");

function paynote_set_webhook() {
    if (isset($_POST['url'])) {
        $url = sanitize_text_field($_POST['url']);
        $gateway = new WC_Gateway_Paynote();
        $result = $gateway->set_webhook($url);

        if (!empty($result) && isset($result->webhook_url)) {
            wp_send_json([
                'success' => true,
                'message' => 'Webhook has been successfully set.',
                'webhook_url' => $result->webhook_url
            ]);
        } else {
            wp_send_json([
                'success' => false,
                'message' => 'Failed to set the webhook.',
                'webhook_url' => null
            ]);
        }
    } else {
        wp_send_json([
            'success' => false,
            'message' => 'Webhook URL is missing.',
            'webhook_url' => null
        ]);
    }
}

add_action("wp_ajax_paynote_get_webhook", "paynote_get_webhook");

function paynote_get_webhook() {
    $gateway = new WC_Gateway_Paynote();
    $result = $gateway->get_webhook();
    if (!empty($result) && isset($result->webhook_url)) {
        wp_send_json([
            'success' => true,
            'webhook_url' => $result->webhook_url
        ]);
    } else {
        wp_send_json([
            'success' => false,
            'webhook_url' => null,
            'message' => 'Unable to retrieve the webhook URL.'
        ]);
    }
}


/*******************************************************/

add_action('admin_enqueue_scripts', 'paynote_js_enqueue');

add_action("wp_ajax_paynote_status_update_all_hook", "paynote_status_update_all");

function paynote_status_update_all() {

    check_ajax_referer('status_update_nonce', 'security');
    paynote_status_update();
}

function paynote_status_update() {
    $gateway = new WC_Gateway_Paynote();
    $args = array(
        'payment_method' => 'paynote',
        'limit' => 100,
        'orderby' => 'date',
        'order' => 'DESC',
        'date_created' => '>' . (time()-30*24*60*60),
        'post_status' => array('on-hold', 'pachp-status', 'processing', 'failed', 'refunded', 'pachoh-status', 'pachd-status', 'pachpr-status'),
    );
    $orders = wc_get_orders($args);
    $options = get_option('paynote_settings');

    foreach ($orders as $order) {

        $order_status = $order->get_status();

        if ($order->get_payment_method() != 'paynote' ||
                $order_status == 'completed' ||
                $order_status == 'cancelled') {
            continue;
        }

        $paynote_check_id = $order->get_meta('paynote_check_id');

        if ($paynote_check_id) {

            $check = $gateway->get_check_by_paynote_check_id($paynote_check_id);

            if ($check) {

                $cancelled = in_array(strtolower($check->status), ["voided", "cancelled"]);
                $processed = in_array(strtolower($check->status), ["printed", "processed"]);
                $refunded = in_array(strtolower($check->status), ["refunded", "refund pending"]);
                $failed = in_array(strtolower($check->status), ["failed", "expired", "refund failed"]);
                $ach_processing = in_array(strtolower($check->status), ["processing"]);
                $ach_declined = in_array(strtolower($check->status), ["declined"]);
                $ach_hold = in_array(strtolower($check->status), ["hold"]);

                if ($cancelled) {
                    if ($order_status != 'cancelled') {
                        $order->add_order_note(__('Check has been cancelled by Paynote Processing.', 'paynote')); 
                        $order->update_status('cancelled');                       
                    }
                } else if ($processed ) {
                    if ($order_status != 'processing' && $order_status != 'completed' && $order_status != 'shipped') {
                        $order->add_order_note(__('Check has been processed by Paynote Processing', 'paynote'));
                        $order->update_status('processing');                                                
                    }
                } else if ($refunded) {
                    if ($order_status != 'refunded') {
                        $order->add_order_note(__('Verification process refunded by Paynote Processing.', 'paynote'));
                        $order->update_status('refunded');                                                
                    }
                } else if ($ach_processing) {
                    if ($order_status != 'pachpr-status') {
                        $order->add_order_note(__('The funds are pending deposit into your account, and the delay depends on your account settings.', 'paynote')); 
                        $order->update_status('pachpr-status');
                    }
                } else if ($ach_declined) {
                    if ($order_status != 'pachd-status') {
                        $order->add_order_note(sprintf(__('The transaction will be declined and not deposited due to - %s', 'paynote'), $check->error_description)); 
                        $order->update_status('pachd-status');                       
                    }
                } else if ($ach_hold) {
                    if ($order_status != 'pachoh-status') {
                        $order->add_order_note(__('The transaction will be placed in a manual review queue and will only be deposited after your manual approval in the dashboard Paynote.', 'paynote')); 
                        $order->update_status('pachoh-status');                       
                    }
                } else if ($failed) {
                    if ($order_status != 'failed') {
                        $order->add_order_note(__('Verification process failed by Paynote Processing.', 'paynote')); 
                        $order->update_status('failed');                       
                    }
                } else {

                    $gateway->log(sprintf(__('Order status: %s, Default order status : %s, Check status: %s ', 'paynote'), $order_status,$options['paynote_settings_default_order_status'],$check->status));

                    if ($order_status != 'on-hold' && $order_status != 'pachp-status'){

                        if ($options['paynote_settings_default_order_status'] == 'on-hold') {
                            $order->add_order_note(__('Verification process completed by Paynote and check is in queue to be processed. Once the check has processed at Paynote, we will update your order status from On-Hold to Processing.', 'paynote')); 
                            $order->update_status('on-hold');
                        }

                        if ($options['paynote_settings_default_order_status'] == 'ach-in-process') {

                            $order->add_order_note(__('Verification process completed by Paynote and check is in queue to be processed. Once the check has processed at Paynote, we will update your order status from ACH In Process to Processing.', 'paynote'));
                            $order->update_status('pachp-status');
                        }
                    }

                }
            } else {
                echo "GATEWAY PAYNOTE PAYMENT ERROR: " . esc_html($gateway->paynote_getLastError());
            }
        }
    } 
}



add_action("wp_ajax_start_session", "paynote_start_session");
add_action("wp_ajax_nopriv_start_session", "paynote_start_session");

function paynote_start_session() {
    $gateway = new WC_Gateway_Paynote();
    $s = ( isset( $_POST["s"] ) ? sanitize_text_field( wp_unslash( $_POST["s"] ) ) : '');
    check_ajax_referer('start_session');

    $options = get_option('paynote_settings');
    $client_id = $options['paynote_test_api_key'];
    $api_pass = $options['paynote_live_api_key'];
    $endpoint = $options['paynote_api_endpoint'];

    if ($client_id && $api_pass && $endpoint) {
        $gateway->log("Attempting to call start_session with id: {$s}.");
        echo esc_html($gateway->start_session($s));
    } else {
        echo false;
    }
}

/**
 * @param array $actions
 */
function paynote_single_status_update_order_action($actions) {
    $actions['paynote_single_status_update_order_action'] = __('Paynote Status Update', 'paynote');
    return $actions;
}

add_action('woocommerce_order_actions', 'paynote_single_status_update_order_action');

/**
 * @param WC_Order $order
 */
function paynote_single_status_update($order) {
    $gateway = new WC_Gateway_Paynote();
    if (
            $order->get_payment_method() != 'paynote' ||
            $order->get_status() == 'failed' ||
            $order->get_status() == 'refunded' ||
            $order->get_status() == 'completed' ||
            $order->get_status() == 'cancelled') {
        return;
    }

    $options = get_option('paynote_settings');
    
    $paynote_check_id = $order->get_meta('paynote_check_id');

    if ($paynote_check_id) {

        $check = $gateway->get_check_by_paynote_check_id($paynote_check_id);

            if ($check) {

                $cancelled = in_array(strtolower($check->status), ["voided", "cancelled"]);
                $processed = in_array(strtolower($check->status), ["printed", "processed"]);
                $refunded = in_array(strtolower($check->status), ["refunded", "refund pending"]);
                $failed = in_array(strtolower($check->status), ["failed", "expired", "refund failed"]);
                $ach_processing = in_array(strtolower($check->status), ["processing"]);
                $ach_declined = in_array(strtolower($check->status), ["declined"]);
                $ach_hold = in_array(strtolower($check->status), ["hold"]);

                $order_status = $order->get_status();

                if ($cancelled) {
                    if ($order_status != 'cancelled') {
                        $order->add_order_note(__('Check has been cancelled by Paynote Processing.', 'paynote'));                        
                        $order->update_status('cancelled');
                    }
                } else if ($processed ) {
                    if ($order_status != 'processing') {
                        $order->add_order_note(__('Check has been processed by Paynote Processing', 'paynote'));                        
                        $order->update_status('processing');
                    }
                } else if ($refunded) {
                    if ($order_status != 'refunded') {
                        $order->add_order_note(__('Verification process refunded by Paynote Processing.', 'paynote'));                        
                        $order->update_status('refunded');
                    }
                } else if ($failed) {
                    if ($order_status != 'failed') {
                        $order->add_order_note(__('Verification process failed by Paynote Processing.', 'paynote'));                        
                        $order->update_status('failed');
                    }
                } else if ($ach_processing) {
                    if ($order_status != 'pachpr-status') {
                        $order->add_order_note(__('The funds are pending deposit into your account, and the delay depends on your account settings.', 'paynote'));                        
                        $order->update_status('pachpr-status');
                    }
                } else if ($ach_declined) {
                    if ($order_status != 'pachd-status') {
                        $order->add_order_note(sprintf(__('The transaction will be declined and not deposited due to - %s', 'paynote'), $check->error_description));                         
                        $order->update_status('pachd-status');
                    }
                } else if ($ach_hold) {
                    if ($order_status != 'pachoh-status') {
                        $order->add_order_note(__('The transaction will be placed in a manual review queue and will only be deposited after your manual approval in the dashboard Paynote.', 'paynote'));                        
                        $order->update_status('pachoh-status');
                    }
                } else {

                    if ($order_status != 'on-hold' && $order_status != 'pachp-status'){

                        if ($options['paynote_settings_default_order_status'] == 'on-hold') {
                            $order->add_order_note(__('Verification process completed by Paynote and check is in queue to be processed. Once the check has processed at Paynote, we will update your order status from On-Hold to Processing.', 'paynote')); 
                            $order->update_status('on-hold');
                        }

                        if ($options['paynote_settings_default_order_status'] == 'ach-in-process') {

                            $order->add_order_note(__('Verification process completed by Paynote and check is in queue to be processed. Once the check has processed at Paynote, we will update your order status from ACH In Process to Processing.', 'paynote'));
                            $order->update_status('pachp-status');
                        }
                    }
                }
            } else {
                echo "GATEWAY PAYNOTE PAYMENT ERROR: " . esc_html($gateway->paynote_getLastError());
            }
    }
}

//END paynote_single_status_update()
add_action('woocommerce_order_action_paynote_single_status_update_order_action', 'paynote_single_status_update');

/**
 * @param string $endpoint  either "Live" or "Test"
 */
function paynote_get_mode($endpoint) {
    if ($endpoint == PAYNOTE_ENDPOINT_LINK_LIVE) {
        $mode = 'Live';
    } else {
        $mode = 'Test';
    }
    return $mode;
}

function paynote_return_schedules_name() {
    $options = get_option('paynote_settings');
    if(isset($options['paynote_settings_order_cron'])){
        $getOrderCronn = $options['paynote_settings_order_cron'];

        switch ($getOrderCronn) {
            case 900:
                return '15min';
                break;
            case 1800:
                return '30min';
                break;
            case 3600:
                return 'everyhours';
                break;
            case 86400:
                return 'everyday';
                break;
            default:
                return '15min';
        }
    }
}

function paynote_cron_reccurences($schedules) {
    if (!isset($schedules["15min"])) {
        $schedules["15min"] = array(
            'interval' => 900,
            'display' => esc_html__('Once every 15 minutes', 'paynote'));
    };
    if (!isset($schedules["30min"])) {
        $schedules["30min"] = array(
            'interval' => 1800,
            'display' => esc_html__('Once every 30 minutes', 'paynote'));
    };
    if (!isset($schedules["everyhours"])) {
        $schedules["everyhours"] = array(
            'interval' => 3600,
            'display' => esc_html__('Once every hours', 'paynote'));
    };
    if (!isset($schedules["everyday"])) {
        $schedules["everyday"] = array(
            'interval' => 86400,
            'display' => esc_html__('Once every day', 'paynote'));
    };
    return $schedules;
}

add_filter('cron_schedules', 'paynote_cron_reccurences');

add_action( 'admin_head', 'paynote_cron_starter' );

add_action('paynote-cron', 'paynote_status_update');

function paynote_cron_starter() {
    $paynoteOptions = get_option('paynote_settings');

    if ($paynoteOptions['paynote_settings_order_cron'] == 0) {
        paynote_remove_event_cron();
    } else {
        paynote_add_event_cron();
    }
}

function getPlaidToken(){
    $options = get_option('paynote_settings');
    $auth_key = $options['paynote_api_endpoint'] == PAYNOTE_ENDPOINT_LINK_LIVE ? $options['paynote_live_api_key'] : $options['paynote_test_api_key'];

    $domenUrl = $options['paynote_api_endpoint'] == PAYNOTE_ENDPOINT_LINK_LIVE ? PAYNOTE_ENDPOINT_LINK_LIVE : PAYNOTE_ENDPOINT_LINK_TEST;
    $request = wp_remote_get( $domenUrl.'api/dwl/customer/plaid/key?key='.$auth_key );
    if( is_wp_error( $request ) ) {
        return '';
    }

    $body = wp_remote_retrieve_body( $request );
    $body_content = json_decode( $body, true );
    return isset($body_content['link_token']['link_token']) ? $body_content['link_token']['link_token'] : '';
}


function paynote_add_event_cron() {
    if (!wp_next_scheduled('paynote-cron')) {
        wp_schedule_event(time(), paynote_return_schedules_name(), 'paynote-cron');
    }
}

register_deactivation_hook(__FILE__, 'paynote_remove_event_cron');

function paynote_remove_event_cron() {
    $timestamp = wp_next_scheduled('paynote-cron');
    wp_unschedule_event($timestamp, 'paynote-cron');
    wp_clear_scheduled_hook('paynote-cron');
}

add_action( 'woocommerce_thankyou', 'paynote_alert_thankyou', 10, 2 ); 

function paynote_alert_thankyou( $order_id ) {
    $order = wc_get_order( $order_id );
    if(isset($_GET['payalert'])){
        echo '<div class="wc-block-components-notices"><div class="wc-block-store-notice wc-block-components-notice-banner is-error is-dismissible"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="24" height="24" aria-hidden="true" focusable="false"><path d="M12 3.2c-4.8 0-8.8 3.9-8.8 8.8 0 4.8 3.9 8.8 8.8 8.8 4.8 0 8.8-3.9 8.8-8.8 0-4.8-4-8.8-8.8-8.8zm0 16c-4 0-7.2-3.3-7.2-7.2C4.8 8 8 4.8 12 4.8s7.2 3.3 7.2 7.2c0 4-3.2 7.2-7.2 7.2zM11 17h2v-6h-2v6zm0-8h2V7h-2v2z"></path></svg><div class="wc-block-components-notice-banner__content"><div>'.$_GET['payalert'].'</div></div></div></div>';
    }
}


add_action('woocommerce_order_status_changed', 'paynote_send_custom_email_notifications', 10, 4);
function paynote_send_custom_email_notifications($order_id, $old_status, $new_status) {
    $order = wc_get_order($order_id);
    $wc_emails = WC()->mailer()->get_emails();
    //delete 
    $gateway = new WC_Gateway_Paynote(); 
    $gateway->log(sprintf(__('Order ID: %s, Payment method : %s, Old status: %s, New status: %s ', 'paynote'), $order_id, $order->get_payment_method(), $old_status, $new_status));


    if(($old_status == 'pachp-status' || $old_status == 'pachoh-status' || $old_status == 'pachd-status' || $old_status == 'pachpr-status') && $new_status == 'processing' && $order->get_payment_method() == 'paynote'){
        //delete 
        $gateway->log(__('WC_Email_Customer_Processing_Order'));
        $wc_emails['WC_Email_Customer_Processing_Order']->trigger($order_id);
    }

    if(($old_status == 'pachp-status' || $old_status == 'pachoh-status' || $old_status == 'pachd-status' || $old_status == 'pachpr-status') && $new_status == 'refunded' && $order->get_payment_method() == 'paynote'){
        //delete 
        $gateway->log(__('WC_Email_Customer_Refunded_Order'));
        $wc_emails['WC_Email_Customer_Refunded_Order']->trigger($order_id);
    }

}

?>
