<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/**
 * @package Paynote
 * @version 1.2.0
 */
/**
 * Plugin Name: Paynote
 * Description: Try Paynote, the easiest way to send and receive bank transfers from clients or vendors. Instant bank verification. Get started today.
 * Author: SeamlessChex
 * Contributors: seamlesschex
 * Version: 1.2.0
 * Author URI: https://www.seamlesschex.com/
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Copyright: © 2025 Paynote
 *
 * Tested up to: 6.7
 * Requires at least: 4.0.3
 * Copyright © 2025 Paynote
 */

define('PAYNOTE_VERSION', '1.2.0');
define('PAYNOTE_VERSION_API', 'v1');

// DEBUG MODE
define('PAYNOTE_DEBUG', FALSE);

define('PAYNOTE_ENDPOINT_LINK_LIVE', 'https://api-paynote.seamlesschex.com/');

if (PAYNOTE_DEBUG) {
    //define('PAYNOTE_ENDPOINT_LINK_TEST', 'https://dev-paynote-wapi.seamlesschex.com/');
    define('PAYNOTE_ENDPOINT_LINK_TEST', 'https://demo-paynote-wapi.seamlesschex.com/');
} else {
    define('PAYNOTE_ENDPOINT_LINK_TEST', 'https://sandbox-paynote.seamlesschex.com/');
}
define('PAYNOTE_PK_KEY', '1902b1de5ce38dfe7586c08873d297');

if (!class_exists('Paynote_Payment_Gateway')) {
    include_once('includes/paynote_extra_functions.php');
    include_once('includes/paynote_settings.php');

    /**
     * class:   Paynote_Payment_Gateway
     * desc:    plugin class to Paynote Payment Gateway
     */
    class Paynote_Payment_Gateway {

        private static $instance;

        public static function instance() {

            if (!self::$instance) {
                self::$instance = new Paynote_Payment_Gateway();
                $networkFlag = False;
                if (in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins')), true)) {
                    $networkFlag = True;
                } else {
                    $active_plugins_var = (array) get_network_option(null, 'active_sitewide_plugins');
                    if (is_array($active_plugins_var) && count($active_plugins_var) != 0 && in_array('woocommerce/woocommerce.php', $active_plugins_var)) {
                        $networkFlag = True;
                    }
                }
                if ($networkFlag) {
                    self::$instance->setup_constants();
                    self::$instance->hooks();
                    self::$instance->includes();
                    self::$instance->load_textdomain();

                    add_filter('woocommerce_payment_gateways', array(self::$instance, 'add_wc_gateway'));
                }
            }

            return self::$instance;
        }

        private function setup_constants() {
            // Plugin path
            define('PAYNOTE_WOO_GSP_DIR', plugin_dir_path(__FILE__));
        }

        private function hooks() {
            register_activation_hook(__FILE__, array('Paynote_Payment_Gateway', 'activate'));
            register_deactivation_hook(__FILE__, array('Paynote_Payment_Gateway', 'deactivate'));
        }

        private function includes() {
            require_once PAYNOTE_WOO_GSP_DIR . 'includes/gateway.php';
        }

        /**
         * Add gateway to WooCommerce.
         *
         * @access public
         * @param  array  $methods
         * @return array
         */
        function add_wc_gateway($methods) {

            $methods[] = 'WC_Gateway_Paynote';
            return $methods;
        }

        public function load_textdomain() {
            // Set filter for language directory
            $lang_dir = PAYNOTE_WOO_GSP_DIR . '/languages/';
            $lang_dir = apply_filters('woo_gateway_paynote_lang_dir', $lang_dir);

            // Traditional WordPress plugin locale filter
            $locale = apply_filters('plugin_locale', get_locale(), '');
            $mofile = sprintf('%1$s-%2$s.mo', 'paynote', $locale);

            // Setup paths to current locale file
            $mofile_local = $lang_dir . $mofile;
            $mofile_global = WP_LANG_DIR . '/paynote/' . $mofile;

            if (file_exists($mofile_global)) {
                // Look in global /wp-content/languages/paynote/ folder
                load_textdomain('paynote', $mofile_global);
            } elseif (file_exists($mofile_local)) {
                // Look in local /wp-content/plugins/paynote/languages/ folder
                load_textdomain('paynote', $mofile_local);
            } else {
                // Load the default language files
                load_plugin_textdomain('paynote', false, $lang_dir);
            }
        }

// END public function __construct()

        public static function activate() {
            flush_rewrite_rules();
        }

        public static function deactivate() {
            flush_rewrite_rules();
        }

    }

    // END class Paynote_Payment_Gateway
}// END if(!class_exists("Paynote_Payment_Gateway"))

function paynote_payment_gateway_load() {

    if (!class_exists('WooCommerce')) {
        require_once( ABSPATH . 'wp-admin/includes/plugin.php' );
        $is_woo = false;
        foreach (get_plugins() as $plugin_path => $plugin) {
            if ('WooCommerce' === $plugin['Name']) {
                $is_woo = true;
                break;
            }
        }
        define('PAYNOTE_HAS_WOO', $is_woo);
        add_action('admin_notices', 'paynote_payment_gateway_notice');
    } else { //else WooCommerce class exists
        return Paynote_Payment_Gateway::instance();
    }
}

add_action( 'activated_plugin', 'paynote_payment_gateway_activated', 10, 2 );

function paynote_payment_gateway_activated(){
    $options = get_option('paynote_settings', null);

            if (!$options) {
                update_option('paynote_settings', [
                    'paynote_settings_order_cron' => 1800,
                    'paynote_gateway_recurring' => 0,
                    'paynote_recurring_cycle' => 'month',
                    'paynote_recurring_installments' => 0,
                    'paynote_save_bank_details' => 0
                ]);
            }
}

add_filter( 'woocommerce_register_shop_order_post_statuses', 'paynote_payment_register_custom_order_status' );

function paynote_payment_register_custom_order_status( $order_statuses ) {
   $options = get_option('paynote_settings');
    
   $order_statuses['wc-pachp-status'] = array(
      'label' => _x( 'ACH In Process', 'Order status', 'paynote' ),
      'public' => true,
      'exclude_from_search' => false,
      'show_in_admin_all_list' => true,
      'show_in_admin_status_list' => true,
      /* translators: %s: number of count */
      'label_count' => _n_noop( 'ACH In Process <span class="count">(%s)</span>', 'ACH In Process <span class="count">(%s)</span>', 'paynote' ),
    );
   $order_statuses['wc-pachpr-status'] = array(
      'label' => _x( 'ACH Processing', 'Order status', 'paynote' ),
      'public' => true,
      'exclude_from_search' => false,
      'show_in_admin_all_list' => true,
      'show_in_admin_status_list' => true,
      /* translators: %s: number of count */
      'label_count' => _n_noop( 'ACH Processing <span class="count">(%s)</span>', 'ACH Processing <span class="count">(%s)</span>', 'paynote' ),
    );
   $order_statuses['wc-pachd-status'] = array(
      'label' => _x( 'ACH Declined', 'Order status', 'paynote' ),
      'public' => true,
      'exclude_from_search' => false,
      'show_in_admin_all_list' => true,
      'show_in_admin_status_list' => true,
      /* translators: %s: number of count */
      'label_count' => _n_noop( 'ACH Declined <span class="count">(%s)</span>', 'ACH Declined <span class="count">(%s)</span>', 'paynote' ),
    );
   $order_statuses['wc-pachoh-status'] = array(
      'label' => _x( 'ACH On hold', 'Order status', 'paynote' ),
      'public' => true,
      'exclude_from_search' => false,
      'show_in_admin_all_list' => true,
      'show_in_admin_status_list' => true,
      /* translators: %s: number of count */
      'label_count' => _n_noop( 'ACH On-Hold <span class="count">(%s)</span>', 'ACH On-Hold <span class="count">(%s)</span>', 'paynote' ),
    );
   return $order_statuses;
}

add_filter( 'wc_order_statuses', 'paynote_payment_show_custom_order_status_single_order_dropdown' );
 
function paynote_payment_show_custom_order_status_single_order_dropdown( $order_statuses ) {
   $options = get_option('paynote_settings');
   $order_statuses['wc-pachp-status'] = _x( 'ACH In Process', 'Order status', 'paynote' );
   $order_statuses['wc-pachpr-status'] = _x( 'ACH Processing', 'Order status', 'paynote' );
   $order_statuses['wc-pachd-status'] = _x( 'ACH Declined', 'Order status', 'paynote' );
   $order_statuses['wc-pachoh-status'] = _x( 'ACH On-Hold', 'Order status', 'paynote' );
   
   return $order_statuses;
}

/* ------------------ Paynote Emails------------------------ */

add_filter( 'woocommerce_email_actions', 'paynote_ach_custom_email_actions', 20, 1 );

function paynote_ach_custom_email_actions( $email_actions ) {
    $email_actions[] = 'woocommerce_order_status_wc-pachp-status';
    $email_actions[] = 'woocommerce_order_status_wc-pachpr-status';
    $email_actions[] = 'woocommerce_order_status_wc-pachd-status';
    $email_actions[] = 'woocommerce_order_status_wc-pachoh-status';    
    return $email_actions;
}

add_filter( 'woocommerce_email_classes', 'paynote_ach_custom_woocommerce_email' );

function paynote_ach_custom_woocommerce_email( $email_classes ) {
    require_once( 'includes/class-wc-custom-email-ach-in-process-order.php' );
    require_once( 'includes/class-wc-custom-email-ach-processing-order.php' );
    require_once( 'includes/class-wc-custom-email-ach-on-hold-order.php' );
    require_once( 'includes/class-wc-custom-email-ach-declined-order.php' );

    $email_classes['WC_Email_Customer_ACH_In_Process_Order'] = new WC_Email_Customer_ACH_In_Process_Order();
       
    $email_classes['WC_Email_Customer_ACH_Processing_Order'] = new WC_Email_Customer_ACH_Processing_Order();
    
    $email_classes['WC_Email_Customer_ACH_On_Hold_Order'] = new WC_Email_Customer_ACH_On_Hold_Order();
   
    $email_classes['WC_Email_Customer_ACH_Declined_Order'] = new WC_Email_Customer_ACH_Declined_Order();
    
    return $email_classes;
}

add_action( 'woocommerce_order_status_pachp-status','send_pachp_status_email' , 10, 10  );
function send_pachp_status_email( $order ) {
    $wc_emails = WC()->mailer()->get_emails();
    $wc_emails['WC_Email_Customer_ACH_In_Process_Order']->trigger($order);
}

add_action( 'woocommerce_order_status_pachoh-status','send_pachoh_status_email' , 10, 10  );
function send_pachoh_status_email( $order ) {
    $wc_emails = WC()->mailer()->get_emails();
    $wc_emails['WC_Email_Customer_ACH_On_Hold_Order']->trigger($order);
}

add_action( 'woocommerce_order_status_pachd-status','send_pachd_status_email' , 10, 10  );
function send_pachd_status_email( $order ) {
    $wc_emails = WC()->mailer()->get_emails();
    $wc_emails['WC_Email_Customer_ACH_Declined_Order']->trigger($order);
}

add_action( 'woocommerce_order_status_pachpr-status','send_pachpr_status_email' , 10, 10  );
function send_pachpr_status_email( $order ) {
    $wc_emails = WC()->mailer()->get_emails();
    $wc_emails['WC_Email_Customer_ACH_Processing_Order']->trigger($order);
}

/* ------------------ and Paynote Emails------------------------ */
 
add_action('plugins_loaded', 'paynote_payment_gateway_load');

function paynote_payment_gateway_notice() {
    if (PAYNOTE_HAS_WOO) {
        echo '<div class="error"><p>' . wp_kses(esc_html__('Paynote CheckoutJS add-on requires WooCommerce! Please activate it to continue!', 'paynote'), $allowed_html_array) . '</p></div>';
    } else {
        echo '<div class="error"><p>' . wp_kses(esc_html__('Paynote CheckoutJS add-on requires WooCommerce! Please install it to continue!', 'paynote'), $allowed_html_array) . '</p></div>';
    }
}

add_action('woocommerce_order_status_cancelled', 'paynote_order_cancelled_so_cancelcheck', 100, 1);

function paynote_order_cancelled_so_cancelcheck($order_id) {
    $gateway = new WC_Gateway_Paynote();
    $gateway->order_cancelled_so_cancelcheck($order_id);
}

add_action('woocommerce_order_refunded', 'paynote_order_refund', 10, 2);
function paynote_order_refund($order_id, $refund_id) {
    $gateway = new WC_Gateway_Paynote();
    $gateway->process_refund($order_id);
}

add_action('wp_enqueue_scripts', 'paynote_blocks_css_js');

function paynote_blocks_css_js() {
    wp_enqueue_style('paynote_blocks_style', plugin_dir_url(__FILE__) . 'css/blocks_style.css', array(), time(), 'all');
    // phpcs:ignore WordPress.WP.EnqueuedResourceParameters.MissingVersion
    wp_enqueue_script('paynote_plaid', 'https://cdn.plaid.com/link/v2/stable/link-initialize.js', false, null, array('strategy' => 'async'));
}

add_action('admin_enqueue_scripts', 'paynote_admin_css');

function paynote_admin_css() {
    wp_enqueue_style('paynote_tooltip', plugin_dir_url(__FILE__) . 'css/tooltip.css', array(), time(), 'all');
    wp_enqueue_style('admin-paynote-style', plugin_dir_url(__FILE__) . 'css/admin_style.css', array(), time(), 'all');
}

add_action( 'woocommerce_api_paynote_webhook', 'paynote_payment_webhook' );

function paynote_payment_webhook() {
    $gateway = new WC_Gateway_Paynote();
    $options = get_option('paynote_settings');

    $auth_key = $options['paynote_api_endpoint'] == PAYNOTE_ENDPOINT_LINK_LIVE ? $options['paynote_live_api_key'] : $options['paynote_test_api_key'];

    if (isset($_SERVER['HTTP_AUTHORIZATION']) && $_SERVER['HTTP_AUTHORIZATION'] == $auth_key) {

        $postdata = json_decode(file_get_contents("php://input"));
        $order_id = preg_replace("!\D!", "", $postdata->check->description);

        $order = wc_get_order($order_id);
        
        if ($order) {

            if (in_array($postdata->event, ["transaction.status", "check.changed", "check.voided", "check.created"])) {

                $paynote_check_id = $order->get_meta('paynote_check_id');
                $order_status = $order->get_status();

                if ($paynote_check_id == $postdata->check->check_id) {

                    $cancelled = in_array(strtolower($postdata->check->status), ["voided", "cancelled"]);
                    $pending = in_array(strtolower($postdata->check->status), ["pending","scheduled"]);
                    $processed = in_array(strtolower($postdata->check->status), ["printed", "processed"]);
                    $refunded = in_array(strtolower($postdata->check->status), ["refunded"]);
                    $failed = in_array(strtolower($postdata->check->status), ["failed", "expired", "refund failed"]);
                    $ach_processing = in_array(strtolower($postdata->check->status), ["processing"]);
                    $ach_declined = in_array(strtolower($postdata->check->status), ["declined"]);
                    $ach_hold = in_array(strtolower($postdata->check->status), ["hold"]);

                    if ($cancelled && $order_status != 'cancelled') {
                        $order->add_order_note(__('Check has been cancelled by Paynote Processing.', 'paynote'));
                        $order->update_status('cancelled');                        
                    } else if ($processed && $order_status != 'processing' && $order_status != 'completed'  && $order_status != 'shipped') {
                        $order->update_status('processing'); 
                        $order->add_order_note(__('Check has been processed by Paynote Processing', 'paynote'));                       
                    } else if ($refunded && $order_status != 'refunded') {
                        $order->add_order_note(__('Verification process refunded by Paynote Processing.', 'paynote'));
                        $order->update_status('refunded');                        
                    } else if ($failed && $order_status != 'failed') {
                        $order->add_order_note(__('Verification process failed by Paynote Processing.', 'paynote')); 
                        $order->update_status('failed');                      
                    } else if ($ach_processing && $order_status != 'pachpr-status') {
                        $order->add_order_note(__('The funds are pending deposit into your account, and the delay depends on your account settings.', 'paynote')); 
                        $order->update_status('pachpr-status');
                        $gateway->log(__('ach_processing in webhook ', 'paynote'));                      
                    } else if ($ach_declined && $order_status != 'pachd-status') {
                        $order->add_order_note(sprintf(__('The transaction will be declined and not deposited due to - %s', 'paynote'), $check->error_description));  
                        $order->update_status('pachd-status');
                        $gateway->log(__('ach_declined in webhook ', 'paynote'));                       
                    } else if ($ach_hold && $order_status != 'pachoh-status') {
                        $order->add_order_note(__('The transaction will be placed in a manual review queue and will only be deposited after your manual approval in the dashboard Paynote.', 'paynote')); 
                        $order->update_status('pachoh-status');
                        $gateway->log(__('ach_hold in webhook ', 'paynote'));                        
                    } else if ($pending) {

                        if ($order_status != 'on-hold' && $order_status != 'pachp-status'){

                            if ($options['paynote_settings_default_order_status'] == 'on-hold') {
                                $order->add_order_note(__('Verification process completed by Paynote and check is in queue to be processed. Once the check has processed at Paynote, we will update your order status from On-Hold to Processing.', 'paynote')); 
                                $order->update_status('on-hold');
                            }

                            if ($options['paynote_settings_default_order_status'] == 'ach-in-process') {

                                $order->add_order_note(__('Verification process completed by Paynote and check is in queue to be processed. Once the check has processed at Paynote, we will update your order status from ACH In Process to Processing.', 'paynote'));
                                $order->update_status('pachp-status');
                                $gateway->log(__('ach-in-process in webhook ', 'paynote'));
                            }
                        }
                    }

                } else if (!$paynote_check_id) {
                    if ($order_status == 'pending') {

                        if ($options['paynote_settings_default_order_status'] == 'on-hold') {
                           $order->update_status('on-hold', esc_html__('Awaiting cheque payment', 'paynote'));
                        } else {
                           $order->update_status('pachp-status', esc_html__('Awaiting cheque payment', 'paynote'));
                        }                        
                        /* translators: %1$s: Check ID, %2$s: Check Number. */
                        $order->add_order_note(sprintf(__('Paynote check accepted (Check_ID: %1$s, CheckNumber: %2$s)', 'paynote'), $postdata->check->check_id, $postdata->check->number));
                        $order->update_meta_data('paynote_check_id', (string) $postdata->check->check_id );
                        $order->update_meta_data('paynote_payment_check_number', (string) $postdata->check->number );
                        $order->save();
                    }
                }
            }
        }

        header('Content-Type: application/json; charset=utf-8');
        exit(wp_json_encode(["order" => $order ? $order->get_data() : '']));
    }

    header("HTTP/1.1 401 Unauthorized");
    exit('{"error":"Authorization fail"}');
}

///////////////////////////////// BLOCKS INTEGRATION //////////////////////////////////////////////////

/**
 * Custom function to declare compatibility with woocommerce_gateway_paynote_blocks feature 
*/

function declare_woocommerce_gateway_paynote_blocks_compatibility() {
    // Check if the required class exists
    if (class_exists('\Automattic\WooCommerce\Utilities\FeaturesUtil')) {
        // Declare compatibility for 'woocommerce_gateway_paynote_blocks'
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('woocommerce_gateway_paynote_blocks', __FILE__, true);
    }
}

// Hook the custom function to the 'before_woocommerce_init' action

add_action('before_woocommerce_init', 'declare_woocommerce_gateway_paynote_blocks_compatibility');

// Hook the custom function to the 'woocommerce_blocks_loaded' action

add_action( 'woocommerce_blocks_loaded', 'paynote_register_order_approval_payment_method_type' );

/**
 * Custom function to register a payment method type
 */

function paynote_register_order_approval_payment_method_type() {
    // Check if the required class exists
    if ( ! class_exists( 'Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType' ) ) {
        return;
    }
    // Include the custom Blocks Checkout class
    require_once plugin_dir_path(__FILE__) . 'wc-gateway-paynote-blocks.php';
    // Hook the registration function to the 'woocommerce_blocks_payment_method_type_registration' action
    add_action(
        'woocommerce_blocks_payment_method_type_registration',
        function( Automattic\WooCommerce\Blocks\Payments\PaymentMethodRegistry $payment_method_registry ) {
            // Register an instance of My_Custom_Gateway_Blocks
            $payment_method_registry->register( new Woocommerce_Gateway_Paynote_Blocks );
        }
    );
}